import { useNavigate } from 'react-router-dom';
import { useServerStore } from '../../stores/serverStore.ts';
import { useVoiceStore } from '../../stores/voiceStore.ts';
import { useAuthStore } from '../../stores/authStore.ts';
import Avatar from '../ui/Avatar.tsx';
import toast from 'react-hot-toast';

export default function ChannelSidebar() {
  const { activeServer, activeChannelId, setActiveChannel } = useServerStore();
  const { joinChannel, currentChannelId, leaveChannel, isMuted, toggleMute, isScreenSharing, screenStream, setScreenStream, setIsScreenSharing } = useVoiceStore();
  const { user, logout } = useAuthStore();
  const navigate = useNavigate();
  const server = activeServer();

  if (!server) return (
    <div style={{ width: 240, background: 'var(--bg-card)', display: 'flex', flexDirection: 'column', borderRight: '1px solid var(--border)', flexShrink: 0 }}>
      <div style={{ padding: 16, borderBottom: '1px solid var(--border)' }}>
        <div style={{ height: 16, width: 120, background: 'rgba(255,255,255,0.06)', borderRadius: 6 }} />
      </div>
      <div style={{ flex: 1, padding: 8, display: 'flex', flexDirection: 'column', gap: 6 }}>
        {[1,2,3].map(i => <div key={i} style={{ height: 32, background: 'rgba(255,255,255,0.04)', borderRadius: 6 }} />)}
      </div>
    </div>
  );

  const textChannels = server.channels.filter((c) => c.type === 'text');
  const voiceChannels = server.channels.filter((c) => c.type === 'voice');
  const activeVoiceChannel = voiceChannels.find((c) => c.id === currentChannelId);

  const startScreenShare = async () => {
    try {
      const stream = await navigator.mediaDevices.getDisplayMedia({ video: true, audio: false });
      setScreenStream(stream);
      setIsScreenSharing(true);
      stream.getVideoTracks()[0].onended = () => stopScreenShare();
    } catch {
      toast.error('Ekran paylaşımı başlatılamadı');
    }
  };

  const stopScreenShare = () => {
    screenStream?.getTracks().forEach((t) => t.stop());
    setScreenStream(null);
    setIsScreenSharing(false);
  };

  return (
    <div style={{
      width: 240, background: 'var(--bg-card)', display: 'flex', flexDirection: 'column',
      borderRight: '1px solid var(--border)', flexShrink: 0,
    }}>
      {/* Server header */}
      <div style={{
        padding: '16px', borderBottom: '1px solid var(--border)',
        display: 'flex', justifyContent: 'space-between', alignItems: 'center',
      }}>
        <h2 style={{ fontSize: 15, fontWeight: 700, overflow: 'hidden', textOverflow: 'ellipsis', whiteSpace: 'nowrap' }}>
          {server.name}
        </h2>
        <i className="fas fa-chevron-down" style={{ color: 'var(--text-muted)', fontSize: 12 }} />
      </div>

      <div style={{ flex: 1, overflowY: 'auto', padding: 8 }}>
        {/* Text channels */}
        <div style={{ marginBottom: 8 }}>
          <div style={{ fontSize: 11, fontWeight: 700, color: 'var(--text-muted)', padding: '4px 8px', letterSpacing: 1, textTransform: 'uppercase' }}>
            Metin Kanalları
          </div>
          {textChannels.map((ch) => (
            <div
              key={ch.id}
              onClick={() => { setActiveChannel(ch.id); navigate(`/servers/${server.id}/channels/${ch.id}`); }}
              className={`sidebar-item ${activeChannelId === ch.id ? 'active' : ''}`}
            >
              <i className="fas fa-hashtag" style={{ fontSize: 14 }} />
              <span>{ch.name}</span>
            </div>
          ))}
        </div>

        {/* Voice channels */}
        {voiceChannels.length > 0 && (
          <div>
            <div style={{ fontSize: 11, fontWeight: 700, color: 'var(--text-muted)', padding: '4px 8px', letterSpacing: 1, textTransform: 'uppercase' }}>
              Sesli Kanallar
            </div>
            {voiceChannels.map((ch) => (
              <div
                key={ch.id}
                onClick={() => joinChannel(ch.id)}
                className={`sidebar-item ${currentChannelId === ch.id ? 'active' : ''}`}
              >
                <i className="fas fa-volume-up" style={{ fontSize: 14 }} />
                <span>{ch.name}</span>
                {currentChannelId === ch.id && (
                  <span style={{ marginLeft: 'auto', fontSize: 10, color: '#22c55e' }}>●</span>
                )}
              </div>
            ))}
          </div>
        )}
      </div>

      {/* Voice status panel - Discord style compact bar */}
      {currentChannelId && activeVoiceChannel && (
        <div style={{
          borderTop: '1px solid var(--border)',
          background: 'rgba(34,197,94,0.06)',
          padding: '8px 10px',
        }}>
          <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', marginBottom: 6 }}>
            <div>
              <div style={{ fontSize: 12, color: '#22c55e', fontWeight: 700 }}>
                <i className="fas fa-circle" style={{ fontSize: 8, marginRight: 5 }} />
                Sesli Bağlantı
              </div>
              <div style={{ fontSize: 11, color: 'var(--text-muted)', marginTop: 1 }}>
                {activeVoiceChannel.name}
              </div>
            </div>
            <button
              onClick={leaveChannel}
              title="Kanaldan ayrıl"
              style={{
                background: 'rgba(239,68,68,0.15)', border: 'none', borderRadius: 6,
                color: '#ef4444', cursor: 'pointer', padding: '4px 8px', fontSize: 13,
              }}
            >
              <i className="fas fa-phone-slash" />
            </button>
          </div>
          <div style={{ display: 'flex', gap: 4 }}>
            <button
              onClick={toggleMute}
              title={isMuted ? 'Sesi aç' : 'Sesi kapat'}
              style={{
                flex: 1, background: isMuted ? 'rgba(239,68,68,0.15)' : 'rgba(255,255,255,0.06)',
                border: 'none', borderRadius: 6, color: isMuted ? '#ef4444' : 'var(--text-muted)',
                cursor: 'pointer', padding: '5px 0', fontSize: 13,
              }}
            >
              <i className={`fas fa-microphone${isMuted ? '-slash' : ''}`} />
            </button>
            <button
              onClick={isScreenSharing ? stopScreenShare : startScreenShare}
              title={isScreenSharing ? 'Ekranı durdur' : 'Ekranı paylaş'}
              style={{
                flex: 1, background: isScreenSharing ? 'rgba(139,92,246,0.2)' : 'rgba(255,255,255,0.06)',
                border: 'none', borderRadius: 6, color: isScreenSharing ? '#8b5cf6' : 'var(--text-muted)',
                cursor: 'pointer', padding: '5px 0', fontSize: 13,
              }}
            >
              <i className="fas fa-desktop" />
            </button>
          </div>
        </div>
      )}

      {/* User panel at bottom */}
      <div style={{
        padding: '10px 12px', borderTop: '1px solid var(--border)',
        display: 'flex', alignItems: 'center', gap: 8,
      }}>
        <Avatar username={user?.username || ''} size={32} status={user?.status} />
        <div style={{ flex: 1, minWidth: 0 }}>
          <div style={{ fontSize: 13, fontWeight: 600, overflow: 'hidden', textOverflow: 'ellipsis' }}>{user?.username}</div>
          <div style={{ fontSize: 11, color: 'var(--text-muted)' }}>#{user?.discriminator}</div>
        </div>
        <button className="btn-ghost" onClick={logout} style={{ padding: '4px 6px', fontSize: 13 }} title="Çıkış">
          <i className="fas fa-sign-out-alt" />
        </button>
      </div>
    </div>
  );
}
