import { useEffect, useRef } from 'react';
import SimplePeer from 'simple-peer';
import { useVoiceStore } from '../stores/voiceStore.ts';
import { useAuthStore } from '../stores/authStore.ts';
import { socket } from './socket.ts';

export function useVoiceConnection() {
  const { currentChannelId, localStream, screenStream, addPeer, removePeer, setLocalStream, setPeerStream, leaveChannel, isScreenSharing } = useVoiceStore();
  const user = useAuthStore((s) => s.user);
  const peersRef = useRef<Record<string, SimplePeer.Instance>>({});
  const localStreamRef = useRef<MediaStream | null>(null);
  const screenStreamRef = useRef<MediaStream | null>(null);

  useEffect(() => { localStreamRef.current = localStream; }, [localStream]);
  useEffect(() => { screenStreamRef.current = screenStream; }, [screenStream]);

  // Handle screen share track add/remove
  useEffect(() => {
    Object.values(peersRef.current).forEach((peer) => {
      if (screenStream) {
        screenStream.getTracks().forEach((track) => {
          try { peer.addTrack(track, screenStream); } catch {}
        });
      } else {
        // Stop screen tracks on peers
        const sender = (peer as any)._pc?.getSenders?.();
        sender?.forEach((s: RTCRtpSender) => {
          if (s.track?.kind === 'video') { try { peer.removeTrack(s.track, screenStream as unknown as MediaStream); } catch {} }
        });
      }
    });
  }, [screenStream]);

  // Init media + join channel
  useEffect(() => {
    if (!currentChannelId || !user) return;
    let mounted = true;

    (async () => {
      try {
        const stream = await navigator.mediaDevices.getUserMedia({ audio: true, video: false });
        if (!mounted) { stream.getTracks().forEach((t) => t.stop()); return; }
        setLocalStream(stream);
        localStreamRef.current = stream;
        socket.emit('voice:join', currentChannelId);
      } catch {
        leaveChannel();
      }
    })();

    return () => {
      mounted = false;
      if (localStreamRef.current) { localStreamRef.current.getTracks().forEach((t) => t.stop()); setLocalStream(null); }
      socket.emit('voice:leave', currentChannelId);
      Object.values(peersRef.current).forEach((p) => p.destroy());
      peersRef.current = {};
    };
  }, [currentChannelId]);

  // Socket listeners
  useEffect(() => {
    if (!currentChannelId || !user) return;

    const onParticipants = ({ users }: { users: string[] }) => {
      users.forEach((id) => { if (id !== user._id) createPeer(id, true); });
    };
    const onSignal = ({ from, signal }: { from: string; signal: unknown }) => {
      const peer = peersRef.current[from];
      if (peer) { peer.signal(signal as SimplePeer.SignalData); }
      else { const p = createPeer(from, false); p.signal(signal as SimplePeer.SignalData); }
    };
    const onUserLeft = ({ userId }: { userId: string }) => {
      peersRef.current[userId]?.destroy();
      delete peersRef.current[userId];
      removePeer(userId);
    };

    socket.on('voice:participants', onParticipants);
    socket.on('voice:signal', onSignal);
    socket.on('voice:user-left', onUserLeft);
    return () => {
      socket.off('voice:participants', onParticipants);
      socket.off('voice:signal', onSignal);
      socket.off('voice:user-left', onUserLeft);
    };
  }, [currentChannelId, user]);

  function createPeer(remoteId: string, initiator: boolean) {
    const streams: MediaStream[] = [];
    if (localStreamRef.current) streams.push(localStreamRef.current);
    if (screenStreamRef.current) streams.push(screenStreamRef.current);

    const peer = new SimplePeer({
      initiator,
      trickle: false,
      stream: localStreamRef.current || undefined,
      config: { iceServers: [{ urls: 'stun:stun.l.google.com:19302' }, { urls: 'stun:stun1.l.google.com:19302' }] },
    });

    if (screenStreamRef.current) {
      screenStreamRef.current.getTracks().forEach((t) => peer.addTrack(t, screenStreamRef.current!));
    }

    peer.on('signal', (signal) => socket.emit('voice:signal', { to: remoteId, signal }));
    peer.on('stream', (stream) => setPeerStream(remoteId, stream));
    peer.on('close', () => { removePeer(remoteId); delete peersRef.current[remoteId]; });
    peer.on('error', () => {});

    peersRef.current[remoteId] = peer;
    addPeer(remoteId, peer);
    return peer;
  }
}
