import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { useServerStore } from '../../stores/serverStore.ts';
import { useAuthStore } from '../../stores/authStore.ts';
import { serverApi } from '../../api/server.api.ts';
import Avatar from '../ui/Avatar.tsx';
import Modal from '../ui/Modal.tsx';
import toast from 'react-hot-toast';

export default function ServerSidebar() {
  const { servers, addServer, setActiveServer, activeServerId } = useServerStore();
  const { user, logout } = useAuthStore();
  const navigate = useNavigate();
  const [showCreate, setShowCreate] = useState(false);
  const [showJoin, setShowJoin] = useState(false);
  const [serverName, setServerName] = useState('');
  const [inviteCode, setInviteCode] = useState('');

  const handleCreate = async () => {
    if (!serverName.trim()) return;
    try {
      const { data } = await serverApi.create(serverName.trim());
      addServer(data);
      setShowCreate(false);
      setServerName('');
      toast.success('Sunucu oluşturuldu!');
    } catch { toast.error('Sunucu oluşturulamadı'); }
  };

  const handleJoin = async () => {
    if (!inviteCode.trim()) return;
    try {
      const { data } = await serverApi.join(inviteCode.trim());
      addServer(data);
      setShowJoin(false);
      setInviteCode('');
      toast.success('Sunucuya katıldınız!');
    } catch { toast.error('Geçersiz davet kodu'); }
  };

  return (
    <div style={{
      width: 72, background: 'var(--bg-card)', display: 'flex', flexDirection: 'column',
      alignItems: 'center', padding: '12px 0', gap: 8, borderRight: '1px solid var(--border)', flexShrink: 0,
    }}>
      {/* DM / Home */}
      <div
        onClick={() => { setActiveServer(null); navigate('/'); }}
        title="Ana Sayfa"
        style={{
          width: 48, height: 48, borderRadius: activeServerId === null ? 16 : '50%',
          background: activeServerId === null ? 'linear-gradient(135deg, #8b5cf6, #06b6d4)' : 'var(--bg-card2)',
          display: 'flex', alignItems: 'center', justifyContent: 'center', cursor: 'pointer',
          transition: 'all 0.2s', fontSize: 20, color: activeServerId === null ? 'white' : 'var(--text-muted)',
        }}
      >
        <i className="fas fa-home" />
      </div>

      <div style={{ width: 32, height: 1, background: 'var(--border)' }} />

      {/* Servers */}
      {servers.map((server) => (
        <div
          key={server.id}
          onClick={() => {
            setActiveServer(server.id);
            const firstText = server.channels.find((c) => c.type === 'text');
            if (firstText) navigate(`/servers/${server.id}/channels/${firstText.id}`);
          }}
          title={server.name}
          style={{
            width: 48, height: 48, borderRadius: activeServerId === server.id ? 16 : '50%',
            background: activeServerId === server.id ? 'linear-gradient(135deg, #8b5cf6, #06b6d4)' : 'var(--bg-card2)',
            display: 'flex', alignItems: 'center', justifyContent: 'center', cursor: 'pointer',
            transition: 'all 0.2s', overflow: 'hidden',
            border: activeServerId === server.id ? '2px solid var(--accent-violet)' : '2px solid transparent',
          }}
        >
          {server.iconUrl ? (
            <img src={server.iconUrl} alt={server.name} style={{ width: '100%', height: '100%', objectFit: 'cover' }} />
          ) : (
            <span style={{ fontWeight: 700, fontSize: 14, color: 'white' }}>
              {server.name.slice(0, 2).toUpperCase()}
            </span>
          )}
        </div>
      ))}

      {/* Add / Join */}
      <button onClick={() => setShowCreate(true)} className="btn-ghost" title="Sunucu Oluştur"
        style={{ width: 48, height: 48, borderRadius: '50%', display: 'flex', alignItems: 'center', justifyContent: 'center', border: '2px dashed var(--border)' }}>
        <i className="fas fa-plus" />
      </button>
      <button onClick={() => setShowJoin(true)} className="btn-ghost" title="Sunucuya Katıl"
        style={{ width: 48, height: 48, borderRadius: '50%', display: 'flex', alignItems: 'center', justifyContent: 'center', border: '2px dashed var(--border)' }}>
        <i className="fas fa-link" />
      </button>

      {/* User avatar at bottom */}
      <div style={{ marginTop: 'auto' }}>
        <Avatar username={user?.username || ''} src={user?.avatarUrl} size={40} status={user?.status} />
      </div>

      {showCreate && (
        <Modal title="Sunucu Oluştur" onClose={() => setShowCreate(false)}>
          <input className="input" placeholder="Sunucu adı" value={serverName} onChange={(e) => setServerName(e.target.value)}
            onKeyDown={(e) => e.key === 'Enter' && handleCreate()} style={{ marginBottom: 16 }} />
          <button className="btn-primary" onClick={handleCreate} style={{ width: '100%' }}>Oluştur</button>
        </Modal>
      )}

      {showJoin && (
        <Modal title="Sunucuya Katıl" onClose={() => setShowJoin(false)}>
          <input className="input" placeholder="Davet kodu" value={inviteCode} onChange={(e) => setInviteCode(e.target.value)}
            onKeyDown={(e) => e.key === 'Enter' && handleJoin()} style={{ marginBottom: 16 }} />
          <button className="btn-primary" onClick={handleJoin} style={{ width: '100%' }}>Katıl</button>
        </Modal>
      )}
    </div>
  );
}
