import { useState } from 'react';
import type { Message } from '@rc/shared';
import Avatar from '../ui/Avatar.tsx';
import { useAuthStore } from '../../stores/authStore.ts';

interface Props {
  message: Message;
  onEdit?: (id: string, content: string) => void;
  onDelete?: (id: string) => void;
}

export default function MessageItem({ message, onEdit, onDelete }: Props) {
  const currentUser = useAuthStore((s) => s.user);
  const [editing, setEditing] = useState(false);
  const [editContent, setEditContent] = useState(message.content);
  const isOwn = message.userId === currentUser?._id;

  const saveEdit = () => {
    if (editContent.trim() && editContent !== message.content) onEdit?.(message.id, editContent.trim());
    setEditing(false);
  };

  return (
    <div className="fade-in" style={{ display: 'flex', gap: 10, padding: '4px 16px', borderRadius: 8, transition: 'background 0.15s' }}
      onMouseEnter={(e) => (e.currentTarget.style.background = 'rgba(255,255,255,0.02)')}
      onMouseLeave={(e) => (e.currentTarget.style.background = 'transparent')}
    >
      <Avatar username={message.user.username} src={message.user.avatarUrl} size={36} />
      <div style={{ flex: 1, minWidth: 0 }}>
        <div style={{ display: 'flex', alignItems: 'baseline', gap: 8, marginBottom: 2 }}>
          <span style={{ fontWeight: 600, fontSize: 14 }}>{message.user.username}</span>
          <span style={{ fontSize: 11, color: 'var(--text-muted)' }}>
            {new Date(message.createdAt).toLocaleTimeString('tr-TR', { hour: '2-digit', minute: '2-digit' })}
          </span>
          {message.edited && <span style={{ fontSize: 10, color: 'var(--text-muted)' }}>(düzenlendi)</span>}
        </div>
        {editing ? (
          <div style={{ display: 'flex', gap: 8 }}>
            <input className="input" value={editContent} onChange={(e) => setEditContent(e.target.value)}
              onKeyDown={(e) => { if (e.key === 'Enter') saveEdit(); if (e.key === 'Escape') setEditing(false); }}
              style={{ flex: 1, padding: '6px 12px' }} autoFocus />
            <button className="btn-primary" onClick={saveEdit} style={{ padding: '6px 12px' }}>Kaydet</button>
            <button className="btn-ghost" onClick={() => setEditing(false)}>İptal</button>
          </div>
        ) : (
          <div style={{ fontSize: 14, color: 'var(--text-primary)', lineHeight: 1.5, wordBreak: 'break-word' }}>
            {message.content}
          </div>
        )}
      </div>
      {isOwn && !editing && (
        <div style={{ display: 'flex', gap: 4, opacity: 0 }} className="msg-actions"
          onMouseEnter={(e) => (e.currentTarget.style.opacity = '1')}
          onMouseLeave={(e) => (e.currentTarget.style.opacity = '0')}
        >
          <button className="btn-ghost" onClick={() => { setEditing(true); setEditContent(message.content); }} style={{ padding: '4px 8px', fontSize: 12 }}>
            <i className="fas fa-pencil" />
          </button>
          <button className="btn-ghost" onClick={() => onDelete?.(message.id)} style={{ padding: '4px 8px', fontSize: 12, color: '#ef4444' }}>
            <i className="fas fa-trash" />
          </button>
        </div>
      )}
    </div>
  );
}
