import { useEffect, useRef } from 'react';
import { useParams } from 'react-router-dom';
import { useMessageStore } from '../stores/messageStore.ts';
import { useServerStore } from '../stores/serverStore.ts';
import { messageApi } from '../api/message.api.ts';
import { socket } from '../socket/socket.ts';
import MessageItem from '../components/message/MessageItem.tsx';
import MessageInput from '../components/message/MessageInput.tsx';
import type { Message } from '@rc/shared';
import toast from 'react-hot-toast';

export default function ChannelPage() {
  const { serverId, channelId } = useParams<{ serverId: string; channelId: string }>();
  const { messages, setMessages, addMessage, updateMessage, removeMessage } = useMessageStore();
  const { activeServer, setActiveServer, setActiveChannel } = useServerStore();
  const bottomRef = useRef<HTMLDivElement>(null);

  const server = activeServer();
  const channel = server?.channels.find((c) => c.id === channelId);
  const msgs = messages[channelId!] || [];

  useEffect(() => {
    if (!serverId || !channelId) return;
    setActiveServer(serverId);
    setActiveChannel(channelId);
    messageApi.list(serverId, channelId).then((r) => setMessages(channelId, [...r.data].reverse())).catch(() => {});
  }, [serverId, channelId]);

  useEffect(() => {
    bottomRef.current?.scrollIntoView({ behavior: 'smooth' });
  }, [msgs.length]);

  useEffect(() => {
    if (!channelId) return;
    socket.emit('voice:join' as any, channelId); // join room for typing events

    const onNew = (msg: Message) => { if (msg.channelId === channelId) addMessage(channelId, msg); };
    const onEdit = (msg: Message) => { if (msg.channelId === channelId) updateMessage(channelId, msg); };
    const onDelete = ({ messageId, channelId: cid }: { messageId: string; channelId: string }) => {
      if (cid === channelId) removeMessage(channelId, messageId);
    };

    socket.on('message:new', onNew);
    socket.on('message:edited', onEdit);
    socket.on('message:deleted', onDelete);
    return () => {
      socket.off('message:new', onNew);
      socket.off('message:edited', onEdit);
      socket.off('message:deleted', onDelete);
    };
  }, [channelId]);

  const handleSend = async (content: string) => {
    if (!serverId || !channelId) return;
    try {
      const { data } = await messageApi.send(serverId, channelId, content);
      addMessage(channelId, data);
      // Broadcast via socket (server also does it, deduplicate if needed)
    } catch { toast.error('Mesaj gönderilemedi'); }
  };

  const handleEdit = async (id: string, content: string) => {
    if (!serverId || !channelId) return;
    try {
      const { data } = await messageApi.edit(serverId, channelId, id, content);
      updateMessage(channelId, data);
    } catch { toast.error('Mesaj düzenlenemedi'); }
  };

  const handleDelete = async (id: string) => {
    if (!serverId || !channelId) return;
    try {
      await messageApi.delete(serverId, channelId, id);
      removeMessage(channelId, id);
    } catch { toast.error('Mesaj silinemedi'); }
  };

  return (
    <div style={{ display: 'flex', flexDirection: 'column', height: '100%', overflow: 'hidden' }}>
      {/* Header */}
      <div style={{ padding: '12px 16px', borderBottom: '1px solid var(--border)', display: 'flex', alignItems: 'center', gap: 10, background: 'var(--bg-card)', flexShrink: 0 }}>
        <i className="fas fa-hashtag" style={{ color: 'var(--text-muted)' }} />
        <span style={{ fontWeight: 700 }}>{channel?.name}</span>
        {channel?.topic && <span style={{ color: 'var(--text-muted)', fontSize: 13 }}>• {channel.topic}</span>}
      </div>

      {/* Messages */}
      <div style={{ flex: 1, overflowY: 'auto', padding: '16px 0', display: 'flex', flexDirection: 'column', gap: 4 }}>
        {msgs.map((msg) => (
          <MessageItem key={msg.id} message={msg} onEdit={handleEdit} onDelete={handleDelete} />
        ))}
        <div ref={bottomRef} />
      </div>

      <MessageInput placeholder={`#${channel?.name ?? 'kanal'}a mesaj yaz`} onSend={handleSend} />
    </div>
  );
}
