import { useEffect, useRef } from 'react';
import { useParams } from 'react-router-dom';
import { dmApi } from '../api/dm.api.ts';
import { socket } from '../socket/socket.ts';
import MessageInput from '../components/message/MessageInput.tsx';
import Avatar from '../components/ui/Avatar.tsx';
import toast from 'react-hot-toast';
import { useState } from 'react';
import type { DirectMessage } from '@rc/shared';
import { useAuthStore } from '../stores/authStore.ts';

export default function DMPage() {
  const { userId } = useParams<{ userId: string }>();
  const [messages, setMessages] = useState<DirectMessage[]>([]);
  const [otherUser, setOtherUser] = useState<DirectMessage['sender'] | null>(null);
  const currentUser = useAuthStore((s) => s.user);
  const bottomRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (!userId) return;
    dmApi.messages(userId).then((r) => {
      const msgs: DirectMessage[] = [...r.data].reverse();
      setMessages(msgs);
      const other = msgs[0]?.senderId !== currentUser?._id ? msgs[0]?.sender : msgs[0]?.receiver;
      if (other) setOtherUser(other);
    }).catch(() => {});
  }, [userId]);

  useEffect(() => {
    bottomRef.current?.scrollIntoView({ behavior: 'smooth' });
  }, [messages.length]);

  useEffect(() => {
    const onDM = (msg: DirectMessage) => {
      if (msg.senderId === userId || msg.receiverId === userId) setMessages((m) => [...m, msg]);
    };
    socket.on('dm:new', onDM);
    return () => { socket.off('dm:new', onDM); };
  }, [userId]);

  const handleSend = async (content: string) => {
    if (!userId) return;
    try {
      const { data } = await dmApi.send(userId, content);
      setMessages((m) => [...m, data]);
    } catch { toast.error('Mesaj gönderilemedi'); }
  };

  return (
    <div style={{ display: 'flex', flexDirection: 'column', height: '100%' }}>
      {/* Header */}
      <div style={{ padding: '12px 16px', borderBottom: '1px solid var(--border)', display: 'flex', alignItems: 'center', gap: 10, background: 'var(--bg-card)', flexShrink: 0 }}>
        {otherUser && <Avatar username={otherUser.username} src={otherUser.avatarUrl} size={32} status={otherUser.status} />}
        <span style={{ fontWeight: 700 }}>{otherUser?.username || 'Direkt Mesaj'}</span>
      </div>

      {/* Messages */}
      <div style={{ flex: 1, overflowY: 'auto', padding: 16, display: 'flex', flexDirection: 'column', gap: 12 }}>
        {messages.map((msg) => {
          const isOwn = msg.senderId === currentUser?._id;
          return (
            <div key={msg.id} style={{ display: 'flex', justifyContent: isOwn ? 'flex-end' : 'flex-start', gap: 8 }}>
              {!isOwn && <Avatar username={msg.sender.username} src={msg.sender.avatarUrl} size={32} />}
              <div style={{
                maxWidth: '70%', padding: '10px 14px', borderRadius: isOwn ? '16px 16px 4px 16px' : '16px 16px 16px 4px',
                background: isOwn ? 'linear-gradient(135deg, #8b5cf6, #06b6d4)' : 'var(--bg-card2)',
                fontSize: 14, lineHeight: 1.5, wordBreak: 'break-word',
              }}>
                {msg.content}
                <div style={{ fontSize: 10, color: isOwn ? 'rgba(255,255,255,0.6)' : 'var(--text-muted)', marginTop: 4, textAlign: 'right' }}>
                  {new Date(msg.createdAt).toLocaleTimeString('tr-TR', { hour: '2-digit', minute: '2-digit' })}
                </div>
              </div>
            </div>
          );
        })}
        <div ref={bottomRef} />
      </div>

      <MessageInput placeholder={`${otherUser?.username || 'kullanıcı'}ya mesaj yaz`} onSend={handleSend} />
    </div>
  );
}
