import { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { friendApi } from '../api/friend.api.ts';
import { useFriendStore } from '../stores/friendStore.ts';
import Avatar from '../components/ui/Avatar.tsx';
import Modal from '../components/ui/Modal.tsx';
import toast from 'react-hot-toast';
import type { User } from '@rc/shared';

export default function FriendsPage() {
  const { friends, pendingRequests, setFriends, setPendingRequests, removeRequest } = useFriendStore();
  const [showAdd, setShowAdd] = useState(false);
  const [addForm, setAddForm] = useState({ username: '', discriminator: '' });
  const [tab, setTab] = useState<'all' | 'pending'>('all');
  const navigate = useNavigate();

  useEffect(() => {
    friendApi.list().then((r) => setFriends(r.data)).catch(() => {});
    friendApi.pending().then((r) => setPendingRequests(r.data)).catch(() => {});
  }, []);

  const handleAdd = async () => {
    try {
      await friendApi.send(addForm.username, addForm.discriminator);
      toast.success('Arkadaşlık isteği gönderildi!');
      setShowAdd(false);
    } catch (e: any) { toast.error(e?.response?.data?.error || 'Gönderilemedi'); }
  };

  const handleAccept = async (id: string) => {
    await friendApi.accept(id);
    removeRequest(id);
    friendApi.list().then((r) => setFriends(r.data));
    toast.success('Arkadaşlık kabul edildi!');
  };

  const handleDecline = async (id: string) => {
    await friendApi.decline(id);
    removeRequest(id);
  };

  return (
    <div style={{ padding: 24, maxWidth: 800, overflow: 'auto', height: '100%' }}>
      {/* Header */}
      <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', marginBottom: 24 }}>
        <div style={{ display: 'flex', gap: 8 }}>
          <button onClick={() => setTab('all')} className={tab === 'all' ? 'btn-primary' : 'btn-ghost'} style={{ padding: '8px 16px' }}>
            Tüm Arkadaşlar <span style={{ marginLeft: 6, background: 'rgba(255,255,255,0.15)', borderRadius: 10, padding: '1px 7px', fontSize: 12 }}>{friends.length}</span>
          </button>
          <button onClick={() => setTab('pending')} className={tab === 'pending' ? 'btn-primary' : 'btn-ghost'} style={{ padding: '8px 16px' }}>
            Bekleyen <span style={{ marginLeft: 6, background: 'rgba(255,255,255,0.15)', borderRadius: 10, padding: '1px 7px', fontSize: 12 }}>{pendingRequests.length}</span>
          </button>
        </div>
        <button className="btn-primary" onClick={() => setShowAdd(true)}>
          <i className="fas fa-user-plus" /> Arkadaş Ekle
        </button>
      </div>

      {/* Friends list */}
      {tab === 'all' && (
        <div style={{ display: 'flex', flexDirection: 'column', gap: 4 }}>
          {friends.length === 0 && <p style={{ color: 'var(--text-muted)', textAlign: 'center', marginTop: 40 }}>Henüz arkadaşın yok.</p>}
          {friends.map((friend: User) => (
            <div key={friend._id} className="glass-card" style={{ display: 'flex', alignItems: 'center', gap: 12, padding: '12px 16px' }}>
              <Avatar username={friend.username} src={friend.avatarUrl} size={40} status={friend.status} />
              <div style={{ flex: 1 }}>
                <div style={{ fontWeight: 600 }}>{friend.username}<span style={{ color: 'var(--text-muted)', fontSize: 13 }}>#{friend.discriminator}</span></div>
                <div style={{ fontSize: 12, color: 'var(--text-muted)', textTransform: 'capitalize' }}>{friend.status}</div>
              </div>
              <button className="btn-ghost" onClick={() => navigate(`/dm/${friend._id}`)} title="Mesaj gönder">
                <i className="fas fa-comment" />
              </button>
            </div>
          ))}
        </div>
      )}

      {tab === 'pending' && (
        <div style={{ display: 'flex', flexDirection: 'column', gap: 4 }}>
          {pendingRequests.length === 0 && <p style={{ color: 'var(--text-muted)', textAlign: 'center', marginTop: 40 }}>Bekleyen istek yok.</p>}
          {pendingRequests.map((req) => (
            <div key={req.id} className="glass-card" style={{ display: 'flex', alignItems: 'center', gap: 12, padding: '12px 16px' }}>
              <Avatar username={req.sender.username} src={req.sender.avatarUrl} size={40} />
              <div style={{ flex: 1 }}>
                <div style={{ fontWeight: 600 }}>{req.sender.username}</div>
                <div style={{ fontSize: 12, color: 'var(--text-muted)' }}>Arkadaşlık isteği gönderdi</div>
              </div>
              <button className="btn-primary" onClick={() => handleAccept(req.id)} style={{ padding: '6px 14px' }}>Kabul</button>
              <button className="btn-danger" onClick={() => handleDecline(req.id)} style={{ padding: '6px 14px' }}>Reddet</button>
            </div>
          ))}
        </div>
      )}

      {showAdd && (
        <Modal title="Arkadaş Ekle" onClose={() => setShowAdd(false)}>
          <p style={{ color: 'var(--text-muted)', fontSize: 14, marginBottom: 16 }}>
            Kullanıcı adı ve discriminator ile arkadaş ekle.
          </p>
          <input className="input" placeholder="Kullanıcı adı" value={addForm.username} onChange={(e) => setAddForm((f) => ({ ...f, username: e.target.value }))} style={{ marginBottom: 12 }} />
          <input className="input" placeholder="Discriminator (örn: 1234)" value={addForm.discriminator} onChange={(e) => setAddForm((f) => ({ ...f, discriminator: e.target.value }))}
            onKeyDown={(e) => e.key === 'Enter' && handleAdd()} style={{ marginBottom: 16 }} />
          <button className="btn-primary" onClick={handleAdd} style={{ width: '100%' }}>İstek Gönder</button>
        </Modal>
      )}
    </div>
  );
}
