import { useEffect, useState } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { serverApi } from '../api/server.api.ts';
import { useServerStore } from '../stores/serverStore.ts';
import { useAuthStore } from '../stores/authStore.ts';
import toast from 'react-hot-toast';

export default function InvitePage() {
  const { code } = useParams<{ code: string }>();
  const navigate = useNavigate();
  const { addServer } = useServerStore();
  const user = useAuthStore((s) => s.user);
  const [preview, setPreview] = useState<{ name: string; iconUrl?: string; memberCount: number } | null>(null);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (!code) return;
    serverApi.preview(code).then((r) => setPreview(r.data)).catch(() => navigate('/'));
  }, [code]);

  const handleJoin = async () => {
    if (!code) return;
    if (!user) { navigate('/login'); return; }
    setLoading(true);
    try {
      const { data } = await serverApi.join(code);
      addServer(data);
      const firstText = data.channels.find((c: any) => c.type === 'text');
      navigate(firstText ? `/servers/${data.id}/channels/${firstText.id}` : '/');
      toast.success(`${data.name} sunucusuna katıldın!`);
    } catch (e: any) {
      toast.error(e?.response?.data?.error || 'Katılım başarısız');
    } finally { setLoading(false); }
  };

  if (!preview) return null;

  return (
    <div style={{ minHeight: '100vh', background: 'var(--bg-primary)', display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
      <div className="glass-card fade-in" style={{ padding: 40, textAlign: 'center', width: 380 }}>
        <div style={{ width: 80, height: 80, borderRadius: '50%', background: 'linear-gradient(135deg, #8b5cf6, #06b6d4)', display: 'flex', alignItems: 'center', justifyContent: 'center', fontSize: 32, fontWeight: 800, color: 'white', margin: '0 auto 16px' }}>
          {preview.name.slice(0, 2).toUpperCase()}
        </div>
        <h2 style={{ fontSize: 22, fontWeight: 700, marginBottom: 8 }}>{preview.name}</h2>
        <p style={{ color: 'var(--text-muted)', fontSize: 14, marginBottom: 24 }}>{preview.memberCount} üye</p>
        <button className="btn-primary" onClick={handleJoin} disabled={loading} style={{ width: '100%', padding: 14 }}>
          {loading ? 'Katılınıyor...' : 'Sunucuya Katıl'}
        </button>
        <button className="btn-ghost" onClick={() => navigate('/')} style={{ width: '100%', marginTop: 8 }}>
          Vazgeç
        </button>
      </div>
    </div>
  );
}
