import { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { authApi } from '../api/auth.api.ts';
import { useAuthStore } from '../stores/authStore.ts';
import { connectSocket } from '../socket/socket.ts';
import toast from 'react-hot-toast';

export default function RegisterPage() {
  const [form, setForm] = useState({ username: '', email: '', password: '', confirm: '' });
  const [loading, setLoading] = useState(false);
  const { setAuth } = useAuthStore();
  const navigate = useNavigate();

  const set = (k: string) => (e: React.ChangeEvent<HTMLInputElement>) => setForm((f) => ({ ...f, [k]: e.target.value }));

  const handleRegister = async (e: React.FormEvent) => {
    e.preventDefault();
    if (form.password !== form.confirm) { toast.error('Şifreler eşleşmiyor'); return; }
    setLoading(true);
    try {
      const { data } = await authApi.register(form.username, form.email, form.password);
      setAuth(data.user, data.accessToken, data.refreshToken);
      connectSocket(data.accessToken);
      navigate('/');
    } catch (err: any) {
      toast.error(err?.response?.data?.error || 'Kayıt başarısız');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div style={{ minHeight: '100vh', background: 'var(--bg-primary)', display: 'flex', alignItems: 'center', justifyContent: 'center', position: 'relative', overflow: 'hidden' }}>
      <div style={{ position: 'absolute', top: '-20%', left: '50%', transform: 'translateX(-50%)', width: 600, height: 600, borderRadius: '50%', background: 'radial-gradient(circle, rgba(6,182,212,0.1) 0%, transparent 70%)', pointerEvents: 'none' }} />

      <div className="glass-card fade-in" style={{ width: 440, padding: 40 }}>
        <div style={{ textAlign: 'center', marginBottom: 32 }}>
          <h1 style={{ fontSize: 32, fontWeight: 800 }}>
            <span className="gradient-text">Rising</span>
            <span style={{ color: 'var(--text-muted)' }}>Chat</span>
          </h1>
          <p style={{ color: 'var(--text-muted)', marginTop: 8, fontSize: 14 }}>Yeni hesap oluştur</p>
        </div>

        <form onSubmit={handleRegister}>
          {[
            { label: 'KULLANICI ADI', key: 'username', type: 'text', placeholder: 'kullanici_adi' },
            { label: 'E-POSTA', key: 'email', type: 'email', placeholder: 'ornek@email.com' },
            { label: 'ŞİFRE', key: 'password', type: 'password', placeholder: '••••••••' },
            { label: 'ŞİFRE TEKRAR', key: 'confirm', type: 'password', placeholder: '••••••••' },
          ].map((f) => (
            <div key={f.key} style={{ marginBottom: 16 }}>
              <label style={{ display: 'block', fontSize: 13, fontWeight: 600, marginBottom: 6, color: 'var(--text-muted)' }}>{f.label}</label>
              <input className="input" type={f.type} value={form[f.key as keyof typeof form]} onChange={set(f.key)} placeholder={f.placeholder} required />
            </div>
          ))}
          <button className="btn-primary" type="submit" disabled={loading} style={{ width: '100%', padding: 14, fontSize: 15, marginTop: 8 }}>
            {loading ? 'Hesap oluşturuluyor...' : 'Kayıt Ol'}
          </button>
        </form>

        <p style={{ textAlign: 'center', marginTop: 20, fontSize: 14, color: 'var(--text-muted)' }}>
          Hesabın var mı?{' '}
          <Link to="/login" style={{ color: 'var(--accent-violet)', fontWeight: 600, textDecoration: 'none' }}>Giriş Yap</Link>
        </p>
      </div>
    </div>
  );
}
