import { create } from 'zustand';
import type SimplePeer from 'simple-peer';

interface VoiceState {
  currentChannelId: string | null;
  localStream: MediaStream | null;
  screenStream: MediaStream | null;
  peers: Record<string, SimplePeer.Instance>;
  peerStreams: Record<string, MediaStream>;
  isScreenSharing: boolean;
  isMuted: boolean;
  joinChannel: (channelId: string) => void;
  leaveChannel: () => void;
  setLocalStream: (stream: MediaStream | null) => void;
  setScreenStream: (stream: MediaStream | null) => void;
  addPeer: (userId: string, peer: SimplePeer.Instance) => void;
  removePeer: (userId: string) => void;
  setPeerStream: (userId: string, stream: MediaStream) => void;
  setIsScreenSharing: (v: boolean) => void;
  toggleMute: () => void;
}

export const useVoiceStore = create<VoiceState>((set, get) => ({
  currentChannelId: null,
  localStream: null,
  screenStream: null,
  peers: {},
  peerStreams: {},
  isScreenSharing: false,
  isMuted: false,
  joinChannel: (channelId) => set({ currentChannelId: channelId }),
  leaveChannel: () => {
    const { localStream, screenStream } = get();
    localStream?.getTracks().forEach((t) => t.stop());
    screenStream?.getTracks().forEach((t) => t.stop());
    set({ currentChannelId: null, localStream: null, screenStream: null, peers: {}, peerStreams: {}, isScreenSharing: false, isMuted: false });
  },
  setLocalStream: (stream) => set({ localStream: stream }),
  setScreenStream: (stream) => set({ screenStream: stream }),
  addPeer: (userId, peer) => set((s) => ({ peers: { ...s.peers, [userId]: peer } })),
  removePeer: (userId) => set((s) => {
    const { [userId]: _, ...peers } = s.peers;
    const { [userId]: __, ...peerStreams } = s.peerStreams;
    return { peers, peerStreams };
  }),
  setPeerStream: (userId, stream) => set((s) => ({ peerStreams: { ...s.peerStreams, [userId]: stream } })),
  setIsScreenSharing: (v) => set({ isScreenSharing: v }),
  toggleMute: () => set((s) => {
    s.localStream?.getAudioTracks().forEach((t) => { t.enabled = s.isMuted; });
    return { isMuted: !s.isMuted };
  }),
}));
