import { prisma } from '../config/prisma.js';
import { AppError } from '../middleware/errorHandler.js';

export async function createChannel(serverId: string, name: string, type: 'text' | 'voice', topic?: string) {
  const sid = parseInt(serverId);
  const server = await prisma.server.findUnique({ where: { id: sid } });
  if (!server) throw new AppError('Server not found', 404);

  const maxPos = await prisma.channel.findFirst({ where: { serverId: sid }, orderBy: { position: 'desc' }, select: { position: true } });
  const position = maxPos ? maxPos.position + 1 : 0;

  return prisma.channel.create({ data: { name: name.toLowerCase().replace(/\s+/g, '-'), type, serverId: sid, topic, position } });
}

export async function getChannels(serverId: string) {
  return prisma.channel.findMany({ where: { serverId: parseInt(serverId) }, orderBy: { position: 'asc' } });
}

export async function getChannel(channelId: string) {
  const channel = await prisma.channel.findUnique({ where: { id: parseInt(channelId) } });
  if (!channel) throw new AppError('Channel not found', 404);
  return channel;
}

export async function updateChannel(channelId: string, updates: { name?: string; topic?: string }) {
  const cid = parseInt(channelId);
  const channel = await prisma.channel.findUnique({ where: { id: cid } });
  if (!channel) throw new AppError('Channel not found', 404);
  return prisma.channel.update({
    where: { id: cid },
    data: {
      ...(updates.name && { name: updates.name.toLowerCase().replace(/\s+/g, '-') }),
      ...(updates.topic !== undefined && { topic: updates.topic }),
    },
  });
}

export async function deleteChannel(channelId: string) {
  const cid = parseInt(channelId);
  const channel = await prisma.channel.findUnique({ where: { id: cid } });
  if (!channel) throw new AppError('Channel not found', 404);
  await prisma.channel.delete({ where: { id: cid } });
  return channel;
}
