import { prisma } from '../config/prisma.js';
import { AppError } from '../middleware/errorHandler.js';

const userSelect = { id: true, username: true, discriminator: true, avatarUrl: true, status: true };

function fmt(m: any) {
  return {
    ...m, id: String(m.id), senderId: String(m.senderId), receiverId: String(m.receiverId),
    sender: { ...m.sender, _id: String(m.sender.id) },
    receiver: { ...m.receiver, _id: String(m.receiver.id) },
  };
}

export async function getConversations(userId: string) {
  const uid = parseInt(userId);
  const dms = await prisma.directMessage.findMany({
    where: { OR: [{ senderId: uid }, { receiverId: uid }] },
    orderBy: { createdAt: 'desc' },
    include: { sender: { select: userSelect }, receiver: { select: userSelect } },
    distinct: ['senderId', 'receiverId'],
  });
  return dms.map(fmt);
}

export async function getMessages(userId: string, otherUserId: string, limit = 50, before?: string) {
  const uid = parseInt(userId), oid = parseInt(otherUserId);
  const msgs = await prisma.directMessage.findMany({
    where: {
      OR: [{ senderId: uid, receiverId: oid }, { senderId: oid, receiverId: uid }],
      ...(before && { id: { lt: parseInt(before) } }),
    },
    orderBy: { createdAt: 'desc' },
    take: limit,
    include: { sender: { select: userSelect }, receiver: { select: userSelect } },
  });
  return msgs.map(fmt);
}

export async function sendMessage(senderId: string, receiverId: string, content: string) {
  const dm = await prisma.directMessage.create({
    data: { content, senderId: parseInt(senderId), receiverId: parseInt(receiverId) },
    include: { sender: { select: userSelect }, receiver: { select: userSelect } },
  });
  return fmt(dm);
}

export async function editMessage(messageId: string, userId: string, content: string) {
  const mid = parseInt(messageId);
  const msg = await prisma.directMessage.findUnique({ where: { id: mid } });
  if (!msg) throw new AppError('Message not found', 404);
  if (msg.senderId !== parseInt(userId)) throw new AppError('You can only edit your own messages', 403);
  const updated = await prisma.directMessage.update({ where: { id: mid }, data: { content }, include: { sender: { select: userSelect }, receiver: { select: userSelect } } });
  return fmt(updated);
}

export async function deleteMessage(messageId: string, userId: string) {
  const mid = parseInt(messageId);
  const msg = await prisma.directMessage.findUnique({ where: { id: mid } });
  if (!msg) throw new AppError('Message not found', 404);
  if (msg.senderId !== parseInt(userId)) throw new AppError('You can only delete your own messages', 403);
  await prisma.directMessage.delete({ where: { id: mid } });
  return { messageId: String(mid) };
}
