import { prisma } from '../config/prisma.js';
import { AppError } from '../middleware/errorHandler.js';

const userSelect = { id: true, username: true, discriminator: true, avatarUrl: true, status: true };

function fmt(m: any) {
  return { ...m, id: String(m.id), channelId: String(m.channelId), userId: String(m.userId), user: { ...m.user, _id: String(m.user.id) } };
}

export async function getMessages(channelId: string, limit = 50, before?: string) {
  const cid = parseInt(channelId);
  const msgs = await prisma.message.findMany({
    where: { channelId: cid, ...(before && { id: { lt: parseInt(before) } }) },
    orderBy: { createdAt: 'desc' },
    take: limit,
    include: { user: { select: userSelect } },
  });
  return msgs.map(fmt);
}

export async function createMessage(channelId: string, authorId: string, content: string) {
  const cid = parseInt(channelId);
  const channel = await prisma.channel.findUnique({ where: { id: cid } });
  if (!channel) throw new AppError('Channel not found', 404);
  const msg = await prisma.message.create({ data: { content, channelId: cid, userId: parseInt(authorId) }, include: { user: { select: userSelect } } });
  return fmt(msg);
}

export async function editMessage(messageId: string, userId: string, content: string) {
  const mid = parseInt(messageId);
  const msg = await prisma.message.findUnique({ where: { id: mid } });
  if (!msg) throw new AppError('Message not found', 404);
  if (msg.userId !== parseInt(userId)) throw new AppError('You can only edit your own messages', 403);
  const updated = await prisma.message.update({ where: { id: mid }, data: { content, edited: true, editedAt: new Date() }, include: { user: { select: userSelect } } });
  return fmt(updated);
}

export async function deleteMessage(messageId: string, userId: string) {
  const mid = parseInt(messageId);
  const msg = await prisma.message.findUnique({ where: { id: mid } });
  if (!msg) throw new AppError('Message not found', 404);
  if (msg.userId !== parseInt(userId)) throw new AppError('You can only delete your own messages', 403);
  await prisma.message.delete({ where: { id: mid } });
  return { messageId: String(mid), channelId: String(msg.channelId) };
}
