import type { Server, Socket } from 'socket.io';

const voiceChannels = new Map<string, Set<string>>();

export function voiceHandler(io: Server, socket: Socket) {
  socket.on('voice:join', (channelId: string) => {
    const userId = socket.data.user.userId;
    if (!voiceChannels.has(channelId)) voiceChannels.set(channelId, new Set());
    const participants = voiceChannels.get(channelId)!;

    socket.emit('voice:participants', { channelId, users: Array.from(participants) });
    participants.add(userId);
    socket.join(`voice:${channelId}`);
    socket.data.voiceChannelId = channelId;
    socket.to(`voice:${channelId}`).emit('voice:user-joined', { userId, username: socket.data.user.username, channelId });
  });

  socket.on('voice:signal', ({ to, signal }: { to: string; signal: unknown }) => {
    io.to(`user:${to}`).emit('voice:signal', { from: socket.data.user.userId, signal });
  });

  socket.on('voice:leave', (channelId: string) => handleLeave(io, socket, channelId));
  socket.on('disconnect', () => { if (socket.data.voiceChannelId) handleLeave(io, socket, socket.data.voiceChannelId); });
}

function handleLeave(io: Server, socket: Socket, channelId: string) {
  const userId = socket.data.user.userId;
  const participants = voiceChannels.get(channelId);
  if (participants) {
    participants.delete(userId);
    if (participants.size === 0) voiceChannels.delete(channelId);
  }
  socket.leave(`voice:${channelId}`);
  socket.data.voiceChannelId = null;
  io.to(`voice:${channelId}`).emit('voice:user-left', { userId, channelId });
}

export function getVoiceParticipants(channelId: string): string[] {
  return Array.from(voiceChannels.get(channelId) || []);
}
