import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import { prisma } from '../config/prisma.js';
import { env } from '../config/env.js';
import { AppError } from '../middleware/errorHandler.js';
import type { User } from '@prisma/client';

function generateDiscriminator(): string {
  return String(Math.floor(Math.random() * 10000)).padStart(4, '0');
}

function generateAccessToken(userId: string): string {
  return jwt.sign({ userId }, env.JWT_SECRET, { expiresIn: '15m' });
}

function generateRefreshToken(userId: string): string {
  return jwt.sign({ userId }, env.JWT_REFRESH_SECRET, { expiresIn: '7d' });
}

export function sanitizeUser(user: User) {
  return {
    _id: String(user.id),
    username: user.username,
    discriminator: user.discriminator,
    email: user.email,
    avatarUrl: user.avatarUrl,
    status: user.status,
    customStatus: user.customStatus,
  };
}

export async function register(username: string, email: string, password: string) {
  const existing = await prisma.user.findUnique({ where: { email } });
  if (existing) throw new AppError('Email already in use', 400);

  let discriminator = '';
  for (let i = 0; i < 100; i++) {
    const d = generateDiscriminator();
    const taken = await prisma.user.findUnique({ where: { username_discriminator: { username, discriminator: d } } });
    if (!taken) { discriminator = d; break; }
  }
  if (!discriminator) throw new AppError('Could not generate unique discriminator. Try a different username.', 400);

  const passwordHash = await bcrypt.hash(password, 12);
  const user = await prisma.user.create({ data: { username, discriminator, email, passwordHash } });

  const accessToken = generateAccessToken(String(user.id));
  const refreshToken = generateRefreshToken(String(user.id));
  await prisma.user.update({ where: { id: user.id }, data: { refreshToken: await bcrypt.hash(refreshToken, 10) } });

  return { accessToken, refreshToken, user: sanitizeUser(user) };
}

export async function login(email: string, password: string) {
  const user = await prisma.user.findUnique({ where: { email } });
  if (!user) throw new AppError('Invalid credentials', 401);

  const match = await bcrypt.compare(password, user.passwordHash);
  if (!match) throw new AppError('Invalid credentials', 401);

  const accessToken = generateAccessToken(String(user.id));
  const refreshToken = generateRefreshToken(String(user.id));
  await prisma.user.update({ where: { id: user.id }, data: { refreshToken: await bcrypt.hash(refreshToken, 10) } });

  return { accessToken, refreshToken, user: sanitizeUser(user) };
}

export async function refreshAccessToken(refreshToken: string) {
  try {
    const decoded = jwt.verify(refreshToken, env.JWT_REFRESH_SECRET) as { userId: string };
    const user = await prisma.user.findUnique({ where: { id: parseInt(decoded.userId) } });
    if (!user?.refreshToken) throw new AppError('Invalid refresh token', 401);
    const valid = await bcrypt.compare(refreshToken, user.refreshToken);
    if (!valid) throw new AppError('Invalid refresh token', 401);
    return { accessToken: generateAccessToken(String(user.id)) };
  } catch (e) {
    if (e instanceof AppError) throw e;
    throw new AppError('Invalid refresh token', 401);
  }
}

export async function logout(userId: string) {
  await prisma.user.update({ where: { id: parseInt(userId) }, data: { refreshToken: null } });
}

export async function getMe(userId: string) {
  const user = await prisma.user.findUnique({ where: { id: parseInt(userId) } });
  if (!user) throw new AppError('User not found', 404);
  return sanitizeUser(user);
}
