import { prisma } from '../config/prisma.js';
import { AppError } from '../middleware/errorHandler.js';

const userSelect = { id: true, username: true, discriminator: true, avatarUrl: true, status: true };

function fmtReq(r: any) {
  return {
    ...r, id: String(r.id), senderId: String(r.senderId), receiverId: String(r.receiverId),
    sender: { ...r.sender, _id: String(r.sender.id) },
    receiver: { ...r.receiver, _id: String(r.receiver.id) },
  };
}

export async function sendRequest(fromUserId: string, username: string, discriminator: string) {
  const fid = parseInt(fromUserId);
  const target = await prisma.user.findUnique({ where: { username_discriminator: { username, discriminator } } });
  if (!target) throw new AppError('User not found', 404);
  if (target.id === fid) throw new AppError('Cannot send friend request to yourself', 400);

  const existing = await prisma.friendRequest.findFirst({
    where: { OR: [{ senderId: fid, receiverId: target.id }, { senderId: target.id, receiverId: fid }], status: 'pending' },
  });
  if (existing) throw new AppError('Friend request already exists', 400);

  const req = await prisma.friendRequest.create({
    data: { senderId: fid, receiverId: target.id },
    include: { sender: { select: userSelect }, receiver: { select: userSelect } },
  });
  return fmtReq(req);
}

export async function acceptRequest(requestId: string, userId: string) {
  const rid = parseInt(requestId), uid = parseInt(userId);
  const req = await prisma.friendRequest.findUnique({ where: { id: rid } });
  if (!req) throw new AppError('Friend request not found', 404);
  if (req.receiverId !== uid) throw new AppError('Not authorized', 403);
  if (req.status !== 'pending') throw new AppError('Request already processed', 400);

  const updated = await prisma.friendRequest.update({
    where: { id: rid },
    data: { status: 'accepted' },
    include: { sender: { select: userSelect }, receiver: { select: userSelect } },
  });
  return fmtReq(updated);
}

export async function declineRequest(requestId: string, userId: string) {
  const rid = parseInt(requestId), uid = parseInt(userId);
  const req = await prisma.friendRequest.findUnique({ where: { id: rid } });
  if (!req) throw new AppError('Friend request not found', 404);
  if (req.receiverId !== uid) throw new AppError('Not authorized', 403);
  await prisma.friendRequest.update({ where: { id: rid }, data: { status: 'declined' } });
}

export async function getFriends(userId: string) {
  const uid = parseInt(userId);
  const accepted = await prisma.friendRequest.findMany({
    where: { OR: [{ senderId: uid }, { receiverId: uid }], status: 'accepted' },
    include: { sender: { select: userSelect }, receiver: { select: userSelect } },
  });
  return accepted.map((r) => {
    const friend = r.senderId === uid ? r.receiver : r.sender;
    return { ...friend, _id: String(friend.id) };
  });
}

export async function getPendingRequests(userId: string) {
  const uid = parseInt(userId);
  const reqs = await prisma.friendRequest.findMany({
    where: { OR: [{ senderId: uid }, { receiverId: uid }], status: 'pending' },
    include: { sender: { select: userSelect }, receiver: { select: userSelect } },
  });
  return reqs.map(fmtReq);
}

export async function removeFriend(userId: string, friendId: string) {
  const uid = parseInt(userId), fid = parseInt(friendId);
  await prisma.friendRequest.deleteMany({
    where: { OR: [{ senderId: uid, receiverId: fid }, { senderId: fid, receiverId: uid }], status: 'accepted' },
  });
}
