import { prisma } from '../config/prisma.js';
import { generateInviteCode } from '../utils/generateInvite.js';
import { AppError } from '../middleware/errorHandler.js';

const userSelect = { id: true, username: true, discriminator: true, avatarUrl: true, status: true };

function formatMember(m: any) {
  return { ...m, userId: String(m.userId), serverId: String(m.serverId), user: { ...m.user, _id: String(m.user.id) } };
}
function formatServer(s: any) {
  return {
    ...s, id: String(s.id), ownerId: String(s.ownerId),
    members: s.members.map(formatMember),
    channels: s.channels.map((c: any) => ({ ...c, id: String(c.id), serverId: String(c.serverId) })),
  };
}

export async function createServer(name: string, userId: string, iconUrl?: string) {
  const uid = parseInt(userId);
  const server = await prisma.server.create({
    data: {
      name, iconUrl, inviteCode: generateInviteCode(), ownerId: uid,
      members: { create: { userId: uid, role: 'admin' } },
      channels: { create: [{ name: 'genel', type: 'text', position: 0 }, { name: 'sesli-kanal', type: 'voice', position: 1 }] },
    },
    include: { members: { include: { user: { select: userSelect } } }, channels: { orderBy: { position: 'asc' } } },
  });
  return formatServer(server);
}

export async function getServer(serverId: string) {
  const server = await prisma.server.findUnique({
    where: { id: parseInt(serverId) },
    include: { members: { include: { user: { select: userSelect } } }, channels: { orderBy: { position: 'asc' } } },
  });
  if (!server) throw new AppError('Server not found', 404);
  return formatServer(server);
}

export async function getUserServers(userId: string) {
  const servers = await prisma.server.findMany({
    where: { members: { some: { userId: parseInt(userId) } } },
    include: { members: { include: { user: { select: userSelect } } }, channels: { orderBy: { position: 'asc' } } },
  });
  return servers.map(formatServer);
}

export async function joinServerByInvite(inviteCode: string, userId: string) {
  const uid = parseInt(userId);
  const server = await prisma.server.findUnique({ where: { inviteCode } });
  if (!server) throw new AppError('Invalid invite code', 404);

  const existing = await prisma.serverMember.findUnique({ where: { serverId_userId: { serverId: server.id, userId: uid } } });
  if (existing) throw new AppError('Already a member', 400);

  await prisma.serverMember.create({ data: { serverId: server.id, userId: uid, role: 'member' } });

  const updated = await prisma.server.findUnique({
    where: { id: server.id },
    include: { members: { include: { user: { select: userSelect } } }, channels: { orderBy: { position: 'asc' } } },
  });
  return formatServer(updated!);
}

export async function getServerPreview(inviteCode: string) {
  const server = await prisma.server.findUnique({ where: { inviteCode }, include: { _count: { select: { members: true } } } });
  if (!server) throw new AppError('Invalid invite code', 404);
  return { name: server.name, iconUrl: server.iconUrl, memberCount: server._count.members };
}

export async function leaveServer(serverId: string, userId: string) {
  const sid = parseInt(serverId), uid = parseInt(userId);
  const server = await prisma.server.findUnique({ where: { id: sid } });
  if (!server) throw new AppError('Server not found', 404);
  if (server.ownerId === uid) throw new AppError('Owner cannot leave the server', 400);
  await prisma.serverMember.delete({ where: { serverId_userId: { serverId: sid, userId: uid } } });
}

export async function deleteServer(serverId: string, userId: string) {
  const sid = parseInt(serverId);
  const server = await prisma.server.findUnique({ where: { id: sid } });
  if (!server) throw new AppError('Server not found', 404);
  if (server.ownerId !== parseInt(userId)) throw new AppError('Only the owner can delete the server', 403);
  await prisma.server.delete({ where: { id: sid } });
}

export async function updateMemberRole(serverId: string, targetUserId: string, newRole: string, requesterId: string) {
  const sid = parseInt(serverId);
  const server = await prisma.server.findUnique({ where: { id: sid } });
  if (!server) throw new AppError('Server not found', 404);
  if (server.ownerId !== parseInt(requesterId)) throw new AppError('Only the owner can change roles', 403);
  const member = await prisma.serverMember.findUnique({ where: { serverId_userId: { serverId: sid, userId: parseInt(targetUserId) } } });
  if (!member) throw new AppError('Member not found', 404);
  return prisma.serverMember.update({ where: { id: member.id }, data: { role: newRole as any } });
}

export async function kickMember(serverId: string, targetUserId: string, requesterId: string) {
  const sid = parseInt(serverId), tid = parseInt(targetUserId);
  const server = await prisma.server.findUnique({ where: { id: sid } });
  if (!server) throw new AppError('Server not found', 404);
  if (server.ownerId === tid) throw new AppError('Cannot kick the server owner', 400);
  const requester = await prisma.serverMember.findUnique({ where: { serverId_userId: { serverId: sid, userId: parseInt(requesterId) } } });
  if (!requester || !['admin', 'moderator'].includes(requester.role)) {
    if (server.ownerId !== parseInt(requesterId)) throw new AppError('Insufficient permissions', 403);
  }
  await prisma.serverMember.delete({ where: { serverId_userId: { serverId: sid, userId: tid } } });
}

export async function regenerateInvite(serverId: string) {
  const newCode = generateInviteCode();
  await prisma.server.update({ where: { id: parseInt(serverId) }, data: { inviteCode: newCode } });
  return newCode;
}
